<?php
// blackbet_api_test.php - Remote API test harness for blackbet1.com
// Creates various POST requests (JSON, form-encoded, multipart) to validate API behavior

function log_line($label, $data) {
    $entry = "[" . date('Y-m-d H:i:s') . "] $label: " . (is_string($data) ? $data : print_r($data, true)) . "\n";
    file_put_contents(__DIR__ . '/api_test_log.txt', $entry, FILE_APPEND);
}

function send_request_json($url, $payload, $extraHeaders = []) {
    $ch = curl_init($url);
    $json = json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    $headers = array_merge([
        'Content-Type: application/json',
        'Accept: application/json'
    ], $extraHeaders);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // test-only
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false); // test-only

    $response = curl_exec($ch);
    $err = curl_error($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return ['status_code' => $status, 'error' => $err, 'response' => $response, 'payload' => $payload];
}

function send_request_form($url, $fields, $extraHeaders = []) {
    $ch = curl_init($url);
    $postFields = http_build_query($fields);
    $headers = array_merge([
        'Content-Type: application/x-www-form-urlencoded; charset=UTF-8',
        'Accept: application/json, */*'
    ], $extraHeaders);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

    $response = curl_exec($ch);
    $err = curl_error($ch);
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return ['status_code' => $status, 'error' => $err, 'response' => $response, 'fields' => $fields];
}

function print_result($title, $result) {
    echo "\n== $title ==\n";
    echo "HTTP: " . $result['status_code'] . "\n";
    if (!empty($result['error'])) {
        echo "cURL Error: " . $result['error'] . "\n";
    }
    echo "Response: " . $result['response'] . "\n";
}

$hostUrl = 'https://blackbet1.com/api/veritral_api.php';
$ipUrl = 'https://13.48.127.118/api/veritral_api.php';
$ipHostHeader = ['Host: blackbet1.com'];

// 1) account_details - JSON (hostname)
$payload1 = [
    'method' => 'account_details',
    'user_id' => 282
];
$r1 = send_request_json($hostUrl, $payload1);
log_line('account_details JSON (host)', $r1);
print_result('account_details JSON (host)', $r1);

// 2) account_details - form-encoded (hostname)
$fields2 = [
    'method' => 'account_details',
    'user_id' => 282
];
$r2 = send_request_form($hostUrl, $fields2);
log_line('account_details FORM (host)', $r2);
print_result('account_details FORM (host)', $r2);

// 3) transaction_bet - JSON (hostname)
$ts = date('YmdHis');
$transactionId = 'api_test_' . $ts;
$roundId = 'round_' . $ts;
$payload3 = [
    'method' => 'transaction_bet',
    'transaction_id' => $transactionId,
    'round_id' => $roundId,
    'user_id' => 282,
    'bet' => 5,
    'game' => 'sports',
    'reason' => [
        'is_offer' => 0,
        'user_id' => 'Agent_Player_282',
        'bets' => [
            [
                'event_id' => 6037811065,
                'match_title' => 'Andorra U21 - Kazakistan U21',
                'market_type' => 'P1XP2',
                'market_name' => 'Maç Sonucu',
                'pick' => ' W1',
                'price' => 1.13
            ],
            [
                'event_id' => 6037809411,
                'match_title' => 'Kuzey İrlanda U21 - Malta U21',
                'market_type' => 'P1XP2',
                'market_name' => 'Maç Sonucu',
                'pick' => ' W1',
                'price' => 1.05
            ]
        ],
        'amount' => 5,
        'transaction_id' => $transactionId,
        'round_id' => $roundId
    ]
];
$r3 = send_request_json($hostUrl, $payload3);
log_line('transaction_bet JSON (host)', $r3);
print_result('transaction_bet JSON (host)', $r3);

// 4) transaction_bet - form-encoded (hostname) with reason JSON as string
$fields4 = [
    'method' => 'transaction_bet',
    'transaction_id' => $transactionId,
    'round_id' => $roundId,
    'user_id' => 282,
    'bet' => 5,
    'game' => 'sports',
    'reason' => json_encode($payload3['reason'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
];
$r4 = send_request_form($hostUrl, $fields4);
log_line('transaction_bet FORM (host)', $r4);
print_result('transaction_bet FORM (host)', $r4);

// 5) Repeat key tests using direct IP with Host header (SNI workaround)
$r5 = send_request_json($ipUrl, $payload1, $ipHostHeader);
log_line('account_details JSON (ip+Host)', $r5);
print_result('account_details JSON (ip+Host)', $r5);

$r6 = send_request_form($ipUrl, $fields2, $ipHostHeader);
log_line('account_details FORM (ip+Host)', $r6);
print_result('account_details FORM (ip+Host)', $r6);

$r7 = send_request_json($ipUrl, $payload3, $ipHostHeader);
log_line('transaction_bet JSON (ip+Host)', $r7);
print_result('transaction_bet JSON (ip+Host)', $r7);

$r8 = send_request_form($ipUrl, $fields4, $ipHostHeader);
log_line('transaction_bet FORM (ip+Host)', $r8);
print_result('transaction_bet FORM (ip+Host)', $r8);

echo "\n-- Testler tamamlandı, detaylar api_test_log.txt dosyasına yazıldı. --\n";