<?php
session_start();
header('Content-Type: application/json');

include '../inc/database.php';

// Kullanıcı giriş kontrolü
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    echo json_encode(['success' => false, 'message' => 'Giriş yapmanız gerekiyor.']);
    exit;
}

$user_id = $_SESSION['user_id'];
$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? '';

try {
    switch ($action) {
        case 'claim':
            $bonus_id = $input['bonus_id'] ?? 0;
            
            // Bonus kontrolü
            $stmt = $pdo->prepare("
                SELECT * FROM dynamic_bonuses 
                WHERE id = ? AND status = 'active' 
                AND (valid_until IS NULL OR valid_until > NOW())
            ");
            $stmt->execute([$bonus_id]);
            $bonus = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$bonus) {
                echo json_encode(['success' => false, 'message' => 'Bonus bulunamadı veya süresi dolmuş.']);
                exit;
            }
            
            // Kullanıcının daha önce bu bonusu alıp almadığını kontrol et
            $stmt = $pdo->prepare("
                SELECT COUNT(*) FROM user_bonus_history 
                WHERE user_id = ? AND bonus_id = ?
            ");
            $stmt->execute([$user_id, $bonus_id]);
            $already_claimed = $stmt->fetchColumn();
            
            if ($already_claimed > 0) {
                echo json_encode(['success' => false, 'message' => 'Bu bonusu daha önce almışsınız.']);
                exit;
            }
            
            // Kullanıcı bilgilerini al
            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                echo json_encode(['success' => false, 'message' => 'Kullanıcı bulunamadı.']);
                exit;
            }
            
            // Bonus şartlarını kontrol et
            $can_claim = true;
            $error_message = '';
            
            if ($bonus['min_deposit'] > 0 && $user['total_deposit'] < $bonus['min_deposit']) {
                $can_claim = false;
                $error_message = 'Minimum yatırım şartını karşılamıyorsunuz.';
            }
            
            if ($bonus['min_bet'] > 0 && $user['total_bet'] < $bonus['min_bet']) {
                $can_claim = false;
                $error_message = 'Minimum bahis şartını karşılamıyorsunuz.';
            }
            
            if (!$can_claim) {
                echo json_encode(['success' => false, 'message' => $error_message]);
                exit;
            }
            
            // Bonus miktarını hesapla
            $bonus_amount = $bonus['bonus_amount'];
            if ($bonus['bonus_type'] == 'percentage') {
                $bonus_amount = ($user['balance'] * $bonus['bonus_amount']) / 100;
                if ($bonus['max_amount'] > 0 && $bonus_amount > $bonus['max_amount']) {
                    $bonus_amount = $bonus['max_amount'];
                }
            }
            
            // Transaction başlat
            $pdo->beginTransaction();
            
            try {
                // Kullanıcının bakiyesini güncelle
                $stmt = $pdo->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
                $stmt->execute([$bonus_amount, $user_id]);
                
                // Bonus geçmişine ekle
                $stmt = $pdo->prepare("
                    INSERT INTO user_bonus_history (user_id, bonus_id, bonus_amount, status, created_at) 
                    VALUES (?, ?, ?, 'completed', NOW())
                ");
                $stmt->execute([$user_id, $bonus_id, $bonus_amount]);
                
                // Transaction'ı tamamla
                $pdo->commit();
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Bonus başarıyla hesabınıza eklendi!',
                    'bonus_amount' => $bonus_amount
                ]);
                
            } catch (Exception $e) {
                $pdo->rollBack();
                echo json_encode(['success' => false, 'message' => 'Bonus eklenirken bir hata oluştu.']);
            }
            break;
            
        case 'claim_seasonal':
            $event_id = $input['event_id'] ?? 0;
            
            // Etkinlik kontrolü
            $stmt = $pdo->prepare("
                SELECT * FROM seasonal_events 
                WHERE id = ? AND status = 'active' 
                AND start_date <= NOW() AND end_date >= NOW()
            ");
            $stmt->execute([$event_id]);
            $event = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$event) {
                echo json_encode(['success' => false, 'message' => 'Etkinlik bulunamadı veya süresi dolmuş.']);
                exit;
            }
            
            // Kullanıcının daha önce bu etkinlik bonusunu alıp almadığını kontrol et
            $stmt = $pdo->prepare("
                SELECT COUNT(*) FROM user_bonus_history 
                WHERE user_id = ? AND bonus_id = ? AND bonus_type = 'seasonal'
            ");
            $stmt->execute([$user_id, $event_id]);
            $already_claimed = $stmt->fetchColumn();
            
            if ($already_claimed > 0) {
                echo json_encode(['success' => false, 'message' => 'Bu etkinlik bonusunu daha önce almışsınız.']);
                exit;
            }
            
            $bonus_amount = $event['bonus_amount'];
            
            // Transaction başlat
            $pdo->beginTransaction();
            
            try {
                // Kullanıcının bakiyesini güncelle
                $stmt = $pdo->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
                $stmt->execute([$bonus_amount, $user_id]);
                
                // Bonus geçmişine ekle
                $stmt = $pdo->prepare("
                    INSERT INTO user_bonus_history (user_id, bonus_id, bonus_amount, bonus_type, status, created_at) 
                    VALUES (?, ?, ?, 'seasonal', 'completed', NOW())
                ");
                $stmt->execute([$user_id, $event_id, $bonus_amount]);
                
                // Transaction'ı tamamla
                $pdo->commit();
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Etkinlik bonusu başarıyla hesabınıza eklendi!',
                    'bonus_amount' => $bonus_amount
                ]);
                
            } catch (Exception $e) {
                $pdo->rollBack();
                echo json_encode(['success' => false, 'message' => 'Bonus eklenirken bir hata oluştu.']);
            }
            break;
            
        case 'get_history':
            $stmt = $pdo->prepare("
                SELECT ubh.*, db.name as bonus_name, db.bonus_type
                FROM user_bonus_history ubh
                LEFT JOIN dynamic_bonuses db ON ubh.bonus_id = db.id
                WHERE ubh.user_id = ?
                ORDER BY ubh.created_at DESC
                LIMIT 50
            ");
            $stmt->execute([$user_id]);
            $history = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'data' => $history]);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Geçersiz işlem.']);
            break;
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Bir hata oluştu: ' . $e->getMessage()]);
}
?>