<?php
session_start();
header('Content-Type: application/json');
require_once '../inc/database.php';

// CORS headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Kullanıcı giriş kontrolü
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    echo json_encode(['success' => false, 'message' => 'Lütfen giriş yapın.']);
    exit;
}

$user_id = $_SESSION['user_id'];
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['action'])) {
    echo json_encode(['success' => false, 'message' => 'Geçersiz istek.']);
    exit;
}

try {
    switch ($input['action']) {
        case 'join':
            if (!isset($input['tournament_id'])) {
                echo json_encode(['success' => false, 'message' => 'Turnuva ID gerekli.']);
                exit;
            }
            
            $tournament_id = (int)$input['tournament_id'];
            
            // Turnuva bilgilerini kontrol et
            $stmt = $pdo->prepare("
                SELECT * FROM tournaments 
                WHERE id = ? AND status = 'active' 
                AND start_date > NOW()
            ");
            $stmt->execute([$tournament_id]);
            $tournament = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$tournament) {
                echo json_encode(['success' => false, 'message' => 'Turnuva bulunamadı veya katılım süresi geçmiş.']);
                exit;
            }
            
            // Zaten katılmış mı kontrol et
            $stmt = $pdo->prepare("
                SELECT COUNT(*) FROM tournament_participants 
                WHERE tournament_id = ? AND user_id = ?
            ");
            $stmt->execute([$tournament_id, $user_id]);
            
            if ($stmt->fetchColumn() > 0) {
                echo json_encode(['success' => false, 'message' => 'Bu turnuvaya zaten katıldınız.']);
                exit;
            }
            
            // Maksimum katılımcı kontrolü
            if ($tournament['max_participants'] > 0) {
                $stmt = $pdo->prepare("
                    SELECT COUNT(*) FROM tournament_participants 
                    WHERE tournament_id = ?
                ");
                $stmt->execute([$tournament_id]);
                
                if ($stmt->fetchColumn() >= $tournament['max_participants']) {
                    echo json_encode(['success' => false, 'message' => 'Turnuva dolu. Maksimum katılımcı sayısına ulaşıldı.']);
                    exit;
                }
            }
            
            // Kullanıcının bakiyesini kontrol et
            if ($tournament['entry_fee'] > 0) {
                $stmt = $pdo->prepare("SELECT balance FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $user_balance = $stmt->fetchColumn();
                
                if ($user_balance < $tournament['entry_fee']) {
                    echo json_encode(['success' => false, 'message' => 'Yetersiz bakiye. Giriş ücreti: ' . number_format($tournament['entry_fee'], 2) . ' TL']);
                    exit;
                }
                
                // Bakiyeden düş
                $stmt = $pdo->prepare("UPDATE users SET balance = balance - ? WHERE id = ?");
                $stmt->execute([$tournament['entry_fee'], $user_id]);
                
                // İşlem kaydı oluştur
                $stmt = $pdo->prepare("
                    INSERT INTO user_transactions (user_id, type, amount, description, created_at) 
                    VALUES (?, 'tournament_fee', ?, ?, NOW())
                ");
                $stmt->execute([$user_id, -$tournament['entry_fee'], 'Turnuva giriş ücreti: ' . $tournament['name']]);
            }
            
            // Turnuvaya katıl
            $stmt = $pdo->prepare("
                INSERT INTO tournament_participants (tournament_id, user_id, joined_at, current_score) 
                VALUES (?, ?, NOW(), 0)
            ");
            $stmt->execute([$tournament_id, $user_id]);
            
            echo json_encode(['success' => true, 'message' => 'Turnuvaya başarıyla katıldınız!']);
            break;
            
        case 'get_leaderboard':
            if (!isset($input['tournament_id'])) {
                echo json_encode(['success' => false, 'message' => 'Turnuva ID gerekli.']);
                exit;
            }
            
            $tournament_id = (int)$input['tournament_id'];
            
            $stmt = $pdo->prepare("
                SELECT 
                    tp.current_score,
                    u.username,
                    tp.joined_at,
                    ROW_NUMBER() OVER (ORDER BY tp.current_score DESC) as rank
                FROM tournament_participants tp
                JOIN users u ON tp.user_id = u.id
                WHERE tp.tournament_id = ?
                ORDER BY tp.current_score DESC
                LIMIT 50
            ");
            $stmt->execute([$tournament_id]);
            $leaderboard = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['success' => true, 'leaderboard' => $leaderboard]);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Geçersiz işlem.']);
            break;
    }
    
} catch (Exception $e) {
    error_log("Tournament API Error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Bir hata oluştu. Lütfen tekrar deneyin.']);
}
?>