// Otomatik şüpheli aktivite tespit monitörü
class SuspiciousActivityMonitor {
    constructor() {
        this.isRunning = false;
        this.interval = null;
        this.checkInterval = 5 * 60 * 1000; // 5 dakikada bir kontrol et
        this.lastCheck = null;
        
        this.init();
    }
    
    init() {
        // Sayfa yüklendiğinde başlat
        this.startMonitoring();
        
        // Sayfa görünürlük değiştiğinde kontrol et
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                this.pauseMonitoring();
            } else {
                this.resumeMonitoring();
            }
        });
        
        // Pencere kapanırken durdur
        window.addEventListener('beforeunload', () => {
            this.stopMonitoring();
        });
    }
    
    startMonitoring() {
        if (this.isRunning) return;
        
        this.isRunning = true;
        console.log('Şüpheli aktivite monitörü başlatıldı');
        
        // İlk kontrolü hemen yap
        this.checkSuspiciousActivity();
        
        // Periyodik kontrolleri başlat
        this.interval = setInterval(() => {
            this.checkSuspiciousActivity();
        }, this.checkInterval);
        
        this.updateStatus('Aktif', 'success');
    }
    
    stopMonitoring() {
        if (!this.isRunning) return;
        
        this.isRunning = false;
        if (this.interval) {
            clearInterval(this.interval);
            this.interval = null;
        }
        
        console.log('Şüpheli aktivite monitörü durduruldu');
        this.updateStatus('Durduruldu', 'danger');
    }
    
    pauseMonitoring() {
        if (this.interval) {
            clearInterval(this.interval);
            this.interval = null;
        }
        this.updateStatus('Duraklatıldı', 'warning');
    }
    
    resumeMonitoring() {
        if (this.isRunning && !this.interval) {
            this.interval = setInterval(() => {
                this.checkSuspiciousActivity();
            }, this.checkInterval);
            this.updateStatus('Aktif', 'success');
        }
    }
    
    async checkSuspiciousActivity() {
        try {
            console.log('Şüpheli aktivite kontrolü yapılıyor...');
            
            const response = await fetch('/api/detect_suspicious_activity.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                }
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.lastCheck = new Date();
                this.updateLastCheck();
                
                if (data.new_detections > 0) {
                    console.log(`${data.new_detections} yeni şüpheli aktivite tespit edildi`);
                    this.showNotification(
                        `${data.new_detections} yeni şüpheli aktivite tespit edildi!`,
                        'warning'
                    );
                    
                    // Sayfayı yenile (eğer suspicious_alerts sayfasındaysak)
                    if (window.location.pathname.includes('suspicious_alerts')) {
                        setTimeout(() => {
                            window.location.reload();
                        }, 2000);
                    }
                } else {
                    console.log('Yeni şüpheli aktivite tespit edilmedi');
                }
                
                this.updateStats(data);
            } else {
                console.error('Şüpheli aktivite kontrolü başarısız:', data.error);
                this.showNotification('Şüpheli aktivite kontrolü başarısız!', 'danger');
            }
            
        } catch (error) {
            console.error('Şüpheli aktivite kontrolü hatası:', error);
            this.showNotification('Bağlantı hatası!', 'danger');
        }
    }
    
    updateStatus(status, type) {
        const statusElement = document.getElementById('monitor-status');
        if (statusElement) {
            statusElement.innerHTML = `
                <span class="badge bg-${type}">${status}</span>
            `;
        }
    }
    
    updateLastCheck() {
        const lastCheckElement = document.getElementById('last-check');
        if (lastCheckElement && this.lastCheck) {
            lastCheckElement.textContent = this.lastCheck.toLocaleString('tr-TR');
        }
    }
    
    updateStats(data) {
        const statsElement = document.getElementById('monitor-stats');
        if (statsElement) {
            statsElement.innerHTML = `
                <small class="text-muted">
                    Son kontrol: ${data.total_users_analyzed} kullanıcı analiz edildi, 
                    ${data.new_detections} yeni tespit
                </small>
            `;
        }
    }
    
    showNotification(message, type) {
        // Bootstrap toast notification
        const toastHtml = `
            <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="fas fa-shield-alt me-2"></i>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        `;
        
        let toastContainer = document.getElementById('toast-container');
        if (!toastContainer) {
            toastContainer = document.createElement('div');
            toastContainer.id = 'toast-container';
            toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
            toastContainer.style.zIndex = '9999';
            document.body.appendChild(toastContainer);
        }
        
        toastContainer.insertAdjacentHTML('beforeend', toastHtml);
        
        const toastElement = toastContainer.lastElementChild;
        const toast = new bootstrap.Toast(toastElement, {
            autohide: true,
            delay: 5000
        });
        
        toast.show();
        
        // Toast kapandıktan sonra DOM'dan kaldır
        toastElement.addEventListener('hidden.bs.toast', () => {
            toastElement.remove();
        });
    }
    
    // Manuel kontrol için
    manualCheck() {
        this.checkSuspiciousActivity();
    }
    
    // Ayarları değiştir
    setCheckInterval(minutes) {
        this.checkInterval = minutes * 60 * 1000;
        if (this.isRunning) {
            this.stopMonitoring();
            this.startMonitoring();
        }
    }
}

// Global instance oluştur
window.suspiciousActivityMonitor = new SuspiciousActivityMonitor();

// Sayfa yüklendiğinde başlat
document.addEventListener('DOMContentLoaded', () => {
    console.log('Şüpheli aktivite monitörü hazır');
});