<?php
session_start(); // Oturum başlat

// MySQLi istisnalarını aç
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

// Basit log fonksiyonu
function logf($msg) {
    // Logları C:\xampp\htdocs\inc\error.log içine yaz
    error_log('[' . date('Y-m-d H:i:s') . '] ' . $msg . PHP_EOL, 3, __DIR__ . '/inc/error.log');
}

// Veritabanı bağlantısı (MySQLi örneği)
$servername = "localhost";
$usernameDb = "blacrhdy_bet1";
$password = "blacrhdy_bet1";
$dbname = "blacrhdy_bet1";

logf('save-bet-slip.php init');

$conn = new mysqli($servername, $usernameDb, $password, $dbname);

// Bağlantı kontrolü
if ($conn->connect_error) {
    logf('DB connect_error: ' . $conn->connect_error);
    die("Bağlantı hatası: " . $conn->connect_error);
}

// Charset ayarla
$conn->set_charset('utf8mb4');

// Bahis yapma işlemi
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Kullanıcının giriş yapıp yapmadığını kontrol et
    $isLoggedIn = isset($_SESSION['loggedin']) && $_SESSION['loggedin'];
    $username = $isLoggedIn ? (string)$_SESSION['username'] : '';
    if (!$isLoggedIn || !$username) {
        logf('Request rejected: user not logged in');
        echo json_encode(['success' => false, 'message' => 'Lütfen giriş yapın.']);
        exit;
    }

    $raw = file_get_contents('php://input');
    $data = json_decode($raw, true);
    logf('save-bet-slip start username=' . $username . ' payload=' . ($raw ?: ''));

    if (!is_array($data)) {
        logf('Invalid JSON payload');
        echo json_encode(['success' => false, 'message' => 'Geçersiz istek gövdesi.']);
        exit;
    }

    $stakeAmount = isset($data['stakeAmount']) ? (float)$data['stakeAmount'] : 0.0; // Kullanıcı bahis miktarı
    $bets = isset($data['bets']) && is_array($data['bets']) ? $data['bets'] : []; // Bahis yapılan maçlar
    $reason = isset($data['reason']) ? (string)$data['reason'] : null; // Opsiyonel reason alanı
    logf('reason param=' . (($reason !== null && $reason !== '') ? $reason : 'NULL'));

    // Kullanıcının bakiyesi
    $userBalance = isset($_SESSION['ana_bakiye']) ? (float)$_SESSION['ana_bakiye'] : null;
    if ($userBalance === null) {
        $stmtBal = $conn->prepare("SELECT ana_bakiye FROM kullanicilar WHERE username = ? LIMIT 1");
        $stmtBal->bind_param("s", $username);
        $stmtBal->execute();
        $resBal = $stmtBal->get_result();
        if ($resBal && $rowBal = $resBal->fetch_assoc()) {
            $userBalance = (float)$rowBal['ana_bakiye'];
            logf('Balance fetched from DB username=' . $username . ' balance=' . $userBalance);
        } else {
            logf('Balance not found for username=' . $username);
            echo json_encode(['success' => false, 'message' => 'Bakiye bilgisi bulunamadı.']);
            exit;
        }
    }

    // Eğer bakiye yetersizse, işlem yapma
    if ($userBalance < $stakeAmount || $stakeAmount <= 0) {
        logf('Insufficient balance: balance=' . $userBalance . ' stake=' . $stakeAmount);
        echo json_encode(['success' => false, 'message' => 'Yetersiz bakiye.']);
        exit;
    }

    if (count($bets) === 0) {
        logf('No bets in payload');
        echo json_encode(['success' => false, 'message' => 'Bahis bulunamadı.']);
        exit;
    }

    // Reason kolonu mevcut mu kontrol et, yoksa eklemeyi dene (opsiyonel)
    $hasReasonColumn = false;
    try {
        $colRes = $conn->query("SHOW COLUMNS FROM bets LIKE 'reason'");
        $hasReasonColumn = ($colRes && $colRes->num_rows > 0);
        if (!$hasReasonColumn) {
            $conn->query("ALTER TABLE bets ADD COLUMN reason VARCHAR(255) NULL");
            $hasReasonColumn = true;
            logf('bets.reason column added');
        }
    } catch (Exception $e) {
        // Sütun eklenemedi ise uygulama çalışmaya devam etsin, bet_description içine reason eklenecek
        logf('Reason column check/alter failed: ' . $e->getMessage());
        $hasReasonColumn = false;
    }

    // Bahis açıklaması (çoklu bahis durumu)
    $betDescription = count($bets) === 1 
        ? 'Tekli bahis' 
        : (string)count($bets) . '’li bahis';

    // Reason kolonu yoksa açıklamaya ekle (görsellik için)
    if (!$hasReasonColumn && $reason) {
        $betDescription .= ' | reason: ' . $reason;
    }

    // Transaction başlat
    $conn->begin_transaction();
    logf('Transaction started');
    try {
        $betInserted = false; // Bahislerin başarıyla eklenip eklenmediğini kontrol etmek için flag

        foreach ($bets as $bet) {
            $eventId = isset($bet['eventId']) ? (int)$bet['eventId'] : 0;
            $eventName = isset($bet['eventName']) ? (string)$bet['eventName'] : '';
            $oddValue = isset($bet['oddValue']) ? (float)$bet['oddValue'] : 0.0;
            $eventDateTime = isset($bet['eventDateTime']) && $bet['eventDateTime'] ? (string)$bet['eventDateTime'] : date('Y-m-d H:i:s');  // event_date (tarih ve saat)
            $selectedOddLabel = isset($bet['selectedOddLabel']) ? (string)$bet['selectedOddLabel'] : '';
            $marketName = isset($bet['marketName']) ? (string)$bet['marketName'] : '';
            $status = isset($bet['status']) && $bet['status'] ? (string)$bet['status'] : 'Bekliyor';  // 'Bekliyor' statüsü

            // event_time alanı NOT NULL olduğu için zaman kısmını ayrıca ekle
            $ts = strtotime($eventDateTime);
            $eventTime = $ts ? date('H:i:s', $ts) : date('H:i:s');

            // `potential_winnings` değerini hesapla
            $potentialWinnings = $stakeAmount * $oddValue;

            // Kullanıcının daha önce bu maça bahis yapıp yapmadığını kontrol et
            $checkBetQuery = "SELECT id FROM bets WHERE username = ? AND event_id = ?";
            $stmt = $conn->prepare($checkBetQuery);
            $stmt->bind_param("si", $username, $eventId);
            $stmt->execute();
            $stmt->store_result();
            logf('Duplicate check event_id=' . $eventId . ' rows=' . $stmt->num_rows);

            if ($stmt->num_rows > 0) {
                // Eğer bahis yapılmışsa, hata mesajı döndür
                logf('Duplicate bet detected for event_id=' . $eventId . ' username=' . $username);
                echo json_encode(['success' => false, 'message' => 'Bu maça daha önce bahis yaptınız.']);
                $conn->rollback();
                exit;
            }

            // Bahisi veritabanına kaydet (bet_date=NOW() ekli)
            if ($hasReasonColumn) {
                $insertBetQuery = "INSERT INTO bets (username, event_id, event_name, odd_value, stake_amount, event_date, event_time, selected_odd_label, market_name, status, bet_description, reason, potential_winnings, bet_date) 
                               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
                $stmt = $conn->prepare($insertBetQuery);
                // Tipler: s i s d d s s s s s s s d (13 param)
                $reasonNonNull = ($reason !== null) ? $reason : '';
                $stmt->bind_param("sisddsssssssd", $username, $eventId, $eventName, $oddValue, $stakeAmount, $eventDateTime, $eventTime, $selectedOddLabel, $marketName, $status, $betDescription, $reasonNonNull, $potentialWinnings);
            } else {
                $insertBetQuery = "INSERT INTO bets (username, event_id, event_name, odd_value, stake_amount, event_date, event_time, selected_odd_label, market_name, status, bet_description, potential_winnings, bet_date) 
                               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
                $stmt = $conn->prepare($insertBetQuery);
                // Tipler: s i s d d s s s s s s d (12 param)
                $stmt->bind_param("sisddssssssd", $username, $eventId, $eventName, $oddValue, $stakeAmount, $eventDateTime, $eventTime, $selectedOddLabel, $marketName, $status, $betDescription, $potentialWinnings);
            }

            $stmt->execute();
            logf('Insert OK for event_id=' . $eventId . ' bet_id=' . $conn->insert_id);
            
            // Bahis başarıyla eklenmişse, flag'i güncelle
            $betInserted = true;
        }

        // Eğer bahisler başarılı şekilde eklenmişse, kullanıcının bakiyesini güncelle
        if ($betInserted) {
            $newBalance = $userBalance - $stakeAmount;
            $updateBalanceQuery = "UPDATE kullanicilar SET ana_bakiye = ? WHERE username = ?";
            $stmt = $conn->prepare($updateBalanceQuery);
            $stmt->bind_param("ds", $newBalance, $username);
            $stmt->execute();
            logf('Balance updated username=' . $username . ' newBalance=' . $newBalance);

            // İşlem başarılıysa, commit işlemi
            $conn->commit();
            logf('Transaction committed');
            echo json_encode(['success' => true, 'message' => 'Bahis başarıyla kaydedildi.']);
        }

    } catch (Exception $e) {
        // Hata durumunda rollback işlemi
        $conn->rollback();
        logf('Transaction rollback error=' . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Bir hata oluştu: ' . $e->getMessage()]);
    }
}

$conn->close();
?>
