<?php
require_once 'auth.php';
require_once 'config.php';

// API endpoint
if (isset($_GET['api']) && $_GET['api'] === 'get_payment_methods') {
    header('Content-Type: application/json; charset=utf-8');
    
    try {
        // Get a random bank for FAST/EFT
        $stmt = $pdo->prepare("SELECT * FROM bankalar ORDER BY RAND() LIMIT 1");
        $stmt->execute();
        $bank = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get crypto addresses
        $crypto_stmt = $pdo->prepare("SELECT * FROM crypto_addresses WHERE network = 'TRC20' ORDER BY RAND() LIMIT 1");
        $crypto_stmt->execute();
        $crypto = $crypto_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get payment method configurations
        $payment_methods = [];
        $methods = ['param', 'paycel', 'hadi', 'ahlpay', 'uption', 'morpara'];
        
        foreach ($methods as $method) {
            $stmt = $pdo->prepare("SELECT * FROM payment_method_configs WHERE method_name = ?");
            $stmt->execute([$method]);
            $config = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $payment_methods[$method] = [
                'name' => ucfirst($method),
                'status' => $config ? ($config['is_active'] ? 'Aktif' : 'Servis Dışı') : 'Servis Dışı',
                'account' => $config ? $config['account_info'] : '',
                'min_amount' => $config ? $config['min_amount'] : 0,
                'max_amount' => $config ? $config['max_amount'] : 0
            ];
        }
        
        $response = [
            'fast' => [
                'name' => 'FAST/EFT',
                'bank_name' => $bank ? $bank['banka_adi'] : 'Banka bulunamadı',
                'account_holder' => $bank ? $bank['adsoyad'] : '',
                'iban' => $bank ? $bank['iban'] : '',
                'account_number' => ''
            ],
            'usdt' => [
                'name' => 'USDT (TRC20)',
                'address' => $crypto ? $crypto['address'] : 'Adres bulunamadı',
                'network' => 'TRC20',
                'note' => $crypto ? $crypto['note'] : ''
            ]
        ];
        
        $response = array_merge($response, $payment_methods);
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
        exit;
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Veritabanı hatası: ' . $e->getMessage()], JSON_UNESCAPED_UNICODE);
        exit;
    }
}

// API endpoint for getting banks
if (isset($_GET['api']) && $_GET['api'] === 'get_banks') {
    header('Content-Type: application/json; charset=utf-8');
    
    try {
        $stmt = $pdo->prepare("SELECT id, banka_adi as bank_name, adsoyad as account_name, iban FROM bankalar ORDER BY banka_adi");
        $stmt->execute();
        $banks = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode($banks, JSON_UNESCAPED_UNICODE);
        exit;
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Veritabanı hatası: ' . $e->getMessage()], JSON_UNESCAPED_UNICODE);
        exit;
    }
}

// Admin check
if (!has_permission('banks_manage')) {
    header('Location: ../index.php');
    exit;
}

// Create necessary tables if not exists
try {
    $pdo->exec("CREATE TABLE IF NOT EXISTS crypto_addresses (
        id INT AUTO_INCREMENT PRIMARY KEY,
        network VARCHAR(50) NOT NULL,
        address TEXT NOT NULL,
        note TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    )");
    
    $pdo->exec("CREATE TABLE IF NOT EXISTS payment_method_configs (
        id INT AUTO_INCREMENT PRIMARY KEY,
        method_name VARCHAR(50) NOT NULL UNIQUE,
        is_active BOOLEAN DEFAULT FALSE,
        account_info TEXT,
        min_amount DECIMAL(10,2) DEFAULT 0,
        max_amount DECIMAL(10,2) DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    )");
    
    // Insert default payment methods if not exists
    $methods = ['param', 'paycel', 'hadi', 'ahlpay', 'uption', 'morpara'];
    foreach ($methods as $method) {
        $stmt = $pdo->prepare("INSERT IGNORE INTO payment_method_configs (method_name) VALUES (?)");
        $stmt->execute([$method]);
    }
} catch (Exception $e) {
    // Table creation failed, but continue
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add_bank':
                try {
                    $stmt = $pdo->prepare("INSERT INTO bankalar (banka_adi, adsoyad, iban) VALUES (?, ?, ?)");
                    $stmt->execute([
                        $_POST['banka_adi'],
                        $_POST['adsoyad'],
                        $_POST['iban']
                    ]);
                    $success_message = "Banka hesabı başarıyla eklendi.";
                } catch (Exception $e) {
                    $error_message = "Hata: " . $e->getMessage();
                }
                break;
                
            case 'update_bank':
                try {
                    $stmt = $pdo->prepare("UPDATE bankalar SET banka_adi = ?, adsoyad = ?, iban = ? WHERE id = ?");
                    $stmt->execute([
                        $_POST['banka_adi'],
                        $_POST['adsoyad'],
                        $_POST['iban'],
                        $_POST['bank_id']
                    ]);
                    $success_message = "Banka hesabı başarıyla güncellendi.";
                } catch (Exception $e) {
                    $error_message = "Hata: " . $e->getMessage();
                }
                break;
                
            case 'delete_bank':
                try {
                    $stmt = $pdo->prepare("DELETE FROM bankalar WHERE id = ?");
                    $stmt->execute([$_POST['bank_id']]);
                    $success_message = "Banka hesabı başarıyla silindi.";
                } catch (Exception $e) {
                    $error_message = "Hata: " . $e->getMessage();
                }
                break;
                
            case 'add_crypto':
                try {
                    $stmt = $pdo->prepare("INSERT INTO crypto_addresses (network, address, note) VALUES (?, ?, ?)");
                    $stmt->execute([
                        $_POST['network'],
                        $_POST['address'],
                        $_POST['note']
                    ]);
                    $success_message = "Kripto adresi başarıyla eklendi.";
                } catch (Exception $e) {
                    $error_message = "Hata: " . $e->getMessage();
                }
                break;
                
            case 'update_crypto':
                try {
                    $stmt = $pdo->prepare("UPDATE crypto_addresses SET network = ?, address = ?, note = ? WHERE id = ?");
                    $stmt->execute([
                        $_POST['network'],
                        $_POST['address'],
                        $_POST['note'],
                        $_POST['crypto_id']
                    ]);
                    $success_message = "Kripto adresi başarıyla güncellendi.";
                } catch (Exception $e) {
                    $error_message = "Hata: " . $e->getMessage();
                }
                break;
                
            case 'delete_crypto':
                try {
                    $stmt = $pdo->prepare("DELETE FROM crypto_addresses WHERE id = ?");
                    $stmt->execute([$_POST['crypto_id']]);
                    $success_message = "Kripto adresi başarıyla silindi.";
                } catch (Exception $e) {
                    $error_message = "Hata: " . $e->getMessage();
                }
                break;
                
            case 'update_payment_method':
                try {
                    $stmt = $pdo->prepare("UPDATE payment_method_configs SET is_active = ?, account_info = ?, min_amount = ?, max_amount = ? WHERE method_name = ?");
                    $stmt->execute([
                        isset($_POST['is_active']) ? 1 : 0,
                        $_POST['account_info'],
                        $_POST['min_amount'],
                        $_POST['max_amount'],
                        $_POST['method_name']
                    ]);
                    $success_message = "Ödeme yöntemi başarıyla güncellendi.";
                } catch (Exception $e) {
                    $error_message = "Hata: " . $e->getMessage();
                }
                break;
        }
    }
}

// Get all banks
$banks_stmt = $pdo->prepare("SELECT * FROM bankalar ORDER BY id DESC");
$banks_stmt->execute();
$banks = $banks_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get all crypto addresses
$crypto_stmt = $pdo->prepare("SELECT * FROM crypto_addresses ORDER BY id DESC");
$crypto_stmt->execute();
$crypto_addresses = $crypto_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get payment method configurations
$payment_configs_stmt = $pdo->prepare("SELECT * FROM payment_method_configs ORDER BY method_name");
$payment_configs_stmt->execute();
$payment_configs = $payment_configs_stmt->fetchAll(PDO::FETCH_ASSOC);

// Banka adları seçenekleri (standart liste)
$bankNameOptions = [
    'Ziraat Bankası',
    'İş Bankası',
    'Garanti BBVA',
    'Yapı Kredi',
    'Akbank',
    'Halkbank',
    'VakıfBank',
    'QNB Finansbank',
    'TEB',
    'ING',
    'Kuveyt Türk',
    'Albaraka',
    'DenizBank',
    'Şekerbank'
];
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ödeme Yöntemleri Yönetimi</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons/font/bootstrap-icons.css" rel="stylesheet">
    <!-- SweetAlert2 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/sweetalert2@11.7.9/dist/sweetalert2.min.css" rel="stylesheet">
    <!-- SweetAlert2 JS -->
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11.7.9/dist/sweetalert2.min.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.3/css/all.min.css">
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <?php include 'sidebar.php'; ?>

            <div class="col-md-10 p-4">
                <?php include __DIR__ . '/topbar.php'; ?>
                <h2 class="mb-4">Ödeme Yöntemleri Yönetimi</h2>
                
                <?php if (isset($success_message)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($success_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <?php if (isset($error_message)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($error_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>
                
                <!-- API Test Section -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-gear"></i> API Test</h5>
                    </div>
                    <div class="card-body">
                        <button class="btn btn-primary" onclick="testAPI()">
                            <i class="bi bi-play-circle"></i> API'yi Test Et
                        </button>
                        <div id="apiResult" class="mt-3" style="display: none;">
                            <div class="bg-light p-3 rounded">
                                <pre id="apiOutput" class="mb-0"></pre>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Ödeme Yöntemleri Konfigürasyonu -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-credit-card"></i> Ödeme Yöntemleri Konfigürasyonu</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead class="table-dark">
                                    <tr>
                                        <th>Yöntem</th>
                                        <th>Durum</th>
                                        <th>Hesap Bilgisi</th>
                                        <th>Min Tutar</th>
                                        <th>Max Tutar</th>
                                        <th>İşlemler</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($payment_configs as $config): ?>
                                    <tr>
                                        <td>
                                            <strong><?= ucfirst(htmlspecialchars($config['method_name'])) ?></strong>
                                        </td>
                                        <td>
                                            <?php if ($config['is_active']): ?>
                                                <span class="badge bg-success">Aktif</span>
                                            <?php else: ?>
                                                <span class="badge bg-danger">Pasif</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= htmlspecialchars($config['account_info']) ?></td>
                                        <td><?= number_format($config['min_amount'], 2) ?> ₺</td>
                                        <td><?= number_format($config['max_amount'], 2) ?> ₺</td>
                                        <td>
                                            <button class="btn btn-sm btn-warning" onclick="editPaymentMethod('<?= htmlspecialchars($config['method_name']) ?>', <?= $config['is_active'] ?>, '<?= htmlspecialchars($config['account_info']) ?>', <?= $config['min_amount'] ?>, <?= $config['max_amount'] ?>)">
                                                <i class="bi bi-pencil"></i> Düzenle
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Banka Hesapları Yönetimi -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-bank"></i> Banka Hesapları Yönetimi</h5>
                    </div>
                    <div class="card-body">
                        <!-- Yeni Banka Ekleme Formu -->
                        <form method="POST" class="mb-4">
                            <input type="hidden" name="action" value="add_bank">
                            <div class="row g-2">
                                <div class="col-md-3">
                                    <label class="form-label">Banka Adı</label>
                                    <select name="banka_adi" class="form-control" required>
                                        <option value="">Banka Seçin</option>
                                        <?php foreach ($bankNameOptions as $bn): ?>
                                            <option value="<?= htmlspecialchars($bn) ?>"><?= htmlspecialchars($bn) ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">Hesap Sahibi</label>
                                    <input type="text" name="adsoyad" class="form-control" placeholder="Hesap Sahibi" required>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">IBAN</label>
                                    <input type="text" name="iban" class="form-control" placeholder="IBAN" required>
                                </div>
                                
                                <div class="col-md-1 d-flex align-items-end">
                                    <button type="submit" class="btn btn-success">
                                        <i class="bi bi-plus"></i> Ekle
                                    </button>
                                </div>
                            </div>
                        </form>
                        
                        <!-- Banka Listesi -->
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead class="table-dark">
                                    <tr>
                                        <th>ID</th>
                                        <th>Banka Adı</th>
                                        <th>Hesap Sahibi</th>
                                        <th>IBAN</th>
                                        
                                        <th>İşlemler</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($banks as $bank): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($bank['id']) ?></td>
                                        <td><?= htmlspecialchars($bank['banka_adi']) ?></td>
                                        <td><?= htmlspecialchars($bank['adsoyad']) ?></td>
                                        <td><?= htmlspecialchars($bank['iban']) ?></td>
                                        
                                        <td>
                                            <button class="btn btn-sm btn-warning" onclick="editBank(<?= $bank['id'] ?>, '<?= htmlspecialchars($bank['banka_adi']) ?>', '<?= htmlspecialchars($bank['adsoyad']) ?>', '<?= htmlspecialchars($bank['iban']) ?>')">
                                                <i class="bi bi-pencil"></i> Düzenle
                                            </button>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="delete_bank">
                                                <input type="hidden" name="bank_id" value="<?= $bank['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Bu bankayı silmek istediğinizden emin misiniz?')">
                                                    <i class="bi bi-trash"></i> Sil
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                
                <!-- Kripto Adresleri Yönetimi -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0"><i class="bi bi-currency-bitcoin"></i> Kripto Adresleri Yönetimi</h5>
                    </div>
                    <div class="card-body">
                        <!-- Yeni Kripto Adresi Ekleme Formu -->
                        <form method="POST" class="mb-4">
                            <input type="hidden" name="action" value="add_crypto">
                            <div class="row g-2">
                                <div class="col-md-2">
                                    <label class="form-label">Ağ</label>
                                    <select name="network" class="form-control" required>
                                        <option value="">Ağ Seçin</option>
                                        <option value="TRC20">TRC20</option>
                                        <option value="ERC20">ERC20</option>
                                        <option value="BEP20">BEP20</option>
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label class="form-label">Kripto Adresi</label>
                                    <input type="text" name="address" class="form-control" placeholder="Kripto Adresi" required>
                                </div>
                                <div class="col-md-4">
                                    <label class="form-label">Not</label>
                                    <input type="text" name="note" class="form-control" placeholder="Not (Opsiyonel)">
                                </div>
                                <div class="col-md-2 d-flex align-items-end">
                                    <button type="submit" class="btn btn-success">
                                        <i class="bi bi-plus"></i> Ekle
                                    </button>
                                </div>
                            </div>
                        </form>
                        
                        <!-- Kripto Adresleri Listesi -->
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead class="table-dark">
                                    <tr>
                                        <th>ID</th>
                                        <th>Ağ</th>
                                        <th>Adres</th>
                                        <th>Not</th>
                                        <th>Oluşturulma</th>
                                        <th>İşlemler</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($crypto_addresses as $crypto): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($crypto['id']) ?></td>
                                        <td><span class="badge bg-info"><?= htmlspecialchars($crypto['network']) ?></span></td>
                                        <td><code><?= htmlspecialchars($crypto['address']) ?></code></td>
                                        <td><?= htmlspecialchars($crypto['note']) ?></td>
                                        <td><?= htmlspecialchars($crypto['created_at']) ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-warning" onclick="editCrypto(<?= $crypto['id'] ?>, '<?= htmlspecialchars($crypto['network']) ?>', '<?= htmlspecialchars($crypto['address']) ?>', '<?= htmlspecialchars($crypto['note']) ?>')">
                                                <i class="bi bi-pencil"></i> Düzenle
                                            </button>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="delete_crypto">
                                                <input type="hidden" name="crypto_id" value="<?= $crypto['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Bu kripto adresini silmek istediğinizden emin misiniz?')">
                                                    <i class="bi bi-trash"></i> Sil
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Ödeme Yöntemi Düzenleme Modal -->
    <div class="modal fade" id="editPaymentMethodModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content" style="background-color: #000; color: #fff;">
                <div class="modal-header" style="background-color: #000; color: #fff; border-bottom: 1px solid #333;">
                    <h5 class="modal-title" style="color: #fff;">Ödeme Yöntemini Düzenle</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body" style="background-color: #000; color: #fff;">
                        <input type="hidden" name="action" value="update_payment_method">
                        <input type="hidden" name="method_name" id="editMethodName">
                        <div class="mb-3">
                            <label class="form-label" style="color: #fff;">Yöntem Adı</label>
                            <input type="text" id="editMethodDisplayName" class="form-control" readonly style="background-color: #333; color: #fff; border: 1px solid #555;">
                        </div>
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="is_active" id="editIsActive">
                                <label class="form-check-label" for="editIsActive" style="color: #fff;">
                                    Aktif
                                </label>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label" style="color: #fff;">Hesap Bilgisi</label>
                            <textarea name="account_info" id="editAccountInfo" class="form-control" rows="3" placeholder="Hesap numarası, kullanıcı adı vb." style="background-color: #333; color: #fff; border: 1px solid #555;"></textarea>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label" style="color: #fff;">Minimum Tutar (₺)</label>
                                    <input type="number" name="min_amount" id="editMinAmount" class="form-control" step="0.01" min="0" style="background-color: #333; color: #fff; border: 1px solid #555;">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label" style="color: #fff;">Maksimum Tutar (₺)</label>
                                    <input type="number" name="max_amount" id="editMaxAmount" class="form-control" step="0.01" min="0" style="background-color: #333; color: #fff; border: 1px solid #555;">
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer" style="background-color: #000; color: #fff; border-top: 1px solid #333;">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                        <button type="submit" class="btn btn-primary">Güncelle</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Banka Düzenleme Modal -->
    <div class="modal fade" id="editBankModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Banka Hesabını Düzenle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="update_bank">
                        <input type="hidden" name="bank_id" id="editBankId">
                        <div class="mb-3">
                            <label class="form-label">Banka Adı</label>
                            <select name="banka_adi" id="editBankName" class="form-control" required>
                                <?php foreach ($bankNameOptions as $bn): ?>
                                    <option value="<?= htmlspecialchars($bn) ?>"><?= htmlspecialchars($bn) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Hesap Sahibi</label>
                            <input type="text" name="adsoyad" id="editAccountHolder" class="form-control" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">IBAN</label>
                            <input type="text" name="iban" id="editIban" class="form-control" required>
                        </div>
                        
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                        <button type="submit" class="btn btn-primary">Güncelle</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Kripto Düzenleme Modal -->
    <div class="modal fade" id="editCryptoModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Kripto Adresini Düzenle</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="update_crypto">
                        <input type="hidden" name="crypto_id" id="editCryptoId">
                        <div class="mb-3">
                            <label class="form-label">Ağ</label>
                            <select name="network" id="editCryptoNetwork" class="form-control" required>
                                <option value="TRC20">TRC20</option>
                                <option value="ERC20">ERC20</option>
                                <option value="BEP20">BEP20</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Adres</label>
                            <input type="text" name="address" id="editCryptoAddress" class="form-control" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Not</label>
                            <input type="text" name="note" id="editCryptoNote" class="form-control">
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">İptal</button>
                        <button type="submit" class="btn btn-primary">Güncelle</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function testAPI() {
            fetch('?api=get_payment_methods')
                .then(response => response.json())
                .then(data => {
                    document.getElementById('apiResult').style.display = 'block';
                    document.getElementById('apiOutput').textContent = JSON.stringify(data, null, 2);
                })
                .catch(error => {
                    document.getElementById('apiResult').style.display = 'block';
                    document.getElementById('apiOutput').textContent = 'Hata: ' + error.message;
                });
        }

        function editPaymentMethod(methodName, isActive, accountInfo, minAmount, maxAmount) {
            document.getElementById('editMethodName').value = methodName;
            document.getElementById('editMethodDisplayName').value = methodName.charAt(0).toUpperCase() + methodName.slice(1);
            document.getElementById('editIsActive').checked = isActive;
            document.getElementById('editAccountInfo').value = accountInfo;
            document.getElementById('editMinAmount').value = minAmount;
            document.getElementById('editMaxAmount').value = maxAmount;
            new bootstrap.Modal(document.getElementById('editPaymentMethodModal')).show();
        }

        function editBank(id, name, holder, iban) {
            document.getElementById('editBankId').value = id;
            const bankSel = document.getElementById('editBankName');
            if (bankSel) {
                let found = false;
                for (let i = 0; i < bankSel.options.length; i++) {
                    if (bankSel.options[i].value === name) { found = true; break; }
                }
                if (!found && name) {
                    const opt = new Option(name, name, true, true);
                    bankSel.add(opt);
                } else {
                    bankSel.value = name;
                }
            }
            document.getElementById('editAccountHolder').value = holder;
            document.getElementById('editIban').value = iban;
            new bootstrap.Modal(document.getElementById('editBankModal')).show();
        }

        function editCrypto(id, network, address, note) {
            document.getElementById('editCryptoId').value = id;
            document.getElementById('editCryptoNetwork').value = network;
            document.getElementById('editCryptoAddress').value = address;
            document.getElementById('editCryptoNote').value = note;
            new bootstrap.Modal(document.getElementById('editCryptoModal')).show();
        }
    </script>
</body>
</html>